<?php
/**
 * Plugin Name: Mosallas.ir - SEO Content Automation
 * Plugin URI: https://mosallas.ir
 * Description: Automatically publish AI-generated SEO articles from Mosallas.ir
 * Version: 1.0.1
 * Author: Mosallas.ir
 * Author URI: https://mosallas.ir
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: mosallas
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MOSALLAS_VERSION', '1.0.0');
define('MOSALLAS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MOSALLAS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MOSALLAS_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-logger.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-api.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-admin.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-publisher.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-database.php';

/**
 * Initialize plugin
 */
function mosallas_init() {
    // Initialize classes
    $admin = new Mosallas_Admin();
    $api = new Mosallas_API();
    $database = new Mosallas_Database();
    
    // Register admin menu
    add_action('admin_menu', array($admin, 'add_admin_menu'));
    
    // Register admin scripts and styles
    add_action('admin_enqueue_scripts', array($admin, 'enqueue_admin_assets'));
    
    // Register REST API endpoints
    add_action('rest_api_init', array($api, 'register_routes'));
    
    // Register AJAX handlers for logs
    add_action('wp_ajax_mosallas_get_logs', array($admin, 'ajax_get_logs'));
    add_action('wp_ajax_mosallas_clear_logs', array($admin, 'ajax_clear_logs'));
    add_action('wp_ajax_mosallas_download_logs', array($admin, 'ajax_download_logs'));
    add_action('wp_ajax_mosallas_update_connection_status', array($admin, 'ajax_update_connection_status'));
    
    // Register AJAX handlers for database sync
    add_action('wp_ajax_mosallas_sync_articles', array($database, 'sync_articles'));
    add_action('wp_ajax_mosallas_sync_products', array($database, 'sync_products'));
    
    // Schedule automatic sync
    add_action('mosallas_auto_sync', array($database, 'sync_articles'));
    add_action('mosallas_auto_sync', array($database, 'sync_products'));
    
    // Hook into post save to sync individual posts
    add_action('save_post', function($post_id) use ($database) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        
        $post = get_post($post_id);
        if ($post && in_array($post->post_status, array('publish', 'private'))) {
            $database->save_article_to_db($post);
        }
    });
    
    // Hook into post save for manual category selection
    add_action('save_post', function($post_id) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        
        $post = get_post($post_id);
        if (!$post || !in_array($post->post_status, array('publish', 'draft'))) {
            return;
        }
        
        // Skip if this is an API-generated post
        if (get_post_meta($post_id, '_mosallas_generated', true)) {
            return;
        }
        
        // Apply LLM-based category selection for manual posts
        mosallas_apply_llm_category_selection($post_id);
    });
    
    // Hook into product save for WooCommerce
    add_action('woocommerce_update_product', function($product_id) use ($database) {
        $product = wc_get_product($product_id);
        if ($product) {
            $database->save_product_to_db($product);
        }
    });
}
add_action('plugins_loaded', 'mosallas_init');

/**
 * Activation hook
 */
function mosallas_activate() {
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Set default options
    if (get_option('mosallas_api_key') === false) {
        add_option('mosallas_api_key', '');
    }
    
    // Set default database options
    if (get_option('mosallas_db_host') === false) {
        add_option('mosallas_db_host', 'localhost');
    }
    if (get_option('mosallas_db_port') === false) {
        add_option('mosallas_db_port', '5432');
    }
    if (get_option('mosallas_db_name') === false) {
        add_option('mosallas_db_name', 'mosallas');
    }
    if (get_option('mosallas_db_user') === false) {
        add_option('mosallas_db_user', '');
    }
    if (get_option('mosallas_db_password') === false) {
        add_option('mosallas_db_password', '');
    }
    if (get_option('mosallas_site_id') === false) {
        add_option('mosallas_site_id', wp_generate_uuid4());
    }
    if (get_option('mosallas_connection_status') === false) {
        add_option('mosallas_connection_status', 'pending');
    }
    if (get_option('mosallas_connection_message') === false) {
        add_option('mosallas_connection_message', __('برای فعال‌سازی ارتباط، روی دکمه «اتصال به وبسایت» کلیک کنید.', 'mosallas'));
    }
    if (get_option('mosallas_connection_updated_at') === false) {
        add_option('mosallas_connection_updated_at', current_time('mysql'));
    }
    
    // Confirm connection if API key exists
    $api_key = get_option('mosallas_api_key');
    if (!empty($api_key)) {
        $api = new Mosallas_API();
        $api->confirm_connection();
    }
    
    // Schedule automatic sync
    $database = new Mosallas_Database();
    $database->schedule_auto_sync();
}
register_activation_hook(__FILE__, 'mosallas_activate');

/**
 * Deactivation hook
 */
function mosallas_deactivate() {
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Unschedule automatic sync
    $database = new Mosallas_Database();
    $database->unschedule_auto_sync();
}
register_deactivation_hook(__FILE__, 'mosallas_deactivate');

/**
 * Uninstall hook
 */
function mosallas_uninstall() {
    // Remove options
    delete_option('mosallas_api_key');
    delete_option('mosallas_connection_status');
    delete_option('mosallas_connection_message');
    delete_option('mosallas_connection_updated_at');
    delete_option('mosallas_connection_error');
    delete_option('mosallas_db_host');
    delete_option('mosallas_db_port');
    delete_option('mosallas_db_name');
    delete_option('mosallas_db_user');
    delete_option('mosallas_db_password');
    delete_option('mosallas_site_id');
}
register_uninstall_hook(__FILE__, 'mosallas_uninstall');

/**
 * Apply LLM-based category selection for manual posts
 */
function mosallas_apply_llm_category_selection($post_id) {
    // Check if API key is configured
    $api_key = get_option('mosallas_api_key');
    if (empty($api_key)) {
        return;
    }
    
    $post = get_post($post_id);
    if (!$post || empty($post->post_title)) {
        return;
    }
    
    // Get all available categories
    $categories = get_categories(array(
        'hide_empty' => false,
        'orderby' => 'name',
        'order' => 'ASC'
    ));
    
    if (empty($categories)) {
        return;
    }
    
    // Prepare categories data for API call
    $categories_data = array();
    foreach ($categories as $category) {
        $categories_data[] = array(
            'id' => $category->term_id,
            'name' => $category->name,
            'description' => $category->description
        );
    }
    
    // Prepare article data
    $article_data = array(
        'title' => $post->post_title,
        'content' => $post->post_content,
        'topic' => $post->post_title // Use title as topic for manual posts
    );
    
    // Call the server API for LLM-based category selection
    $selected_category = mosallas_select_category_with_llm($categories_data, $article_data);
    
    if ($selected_category && isset($selected_category['categoryId'])) {
        // Update post categories
        wp_set_post_categories($post_id, array($selected_category['categoryId']));
        
        // Log the selection
        Mosallas_Logger::info('🤖 دسته‌بندی دستی انتخاب شد توسط هوش مصنوعی', array(
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'category_id' => $selected_category['categoryId'],
            'category_name' => $selected_category['categoryName'],
            'reasoning' => $selected_category['reasoning']
        ));
        
        // Add a notice for the user
        add_action('admin_notices', function() use ($selected_category) {
            echo '<div class="notice notice-success is-dismissible"><p>';
            echo 'دسته‌بندی "' . esc_html($selected_category['categoryName']) . '" توسط هوش مصنوعی انتخاب شد.';
            echo '</p></div>';
        });
    }
}

/**
 * Select category using LLM via server API
 */
function mosallas_select_category_with_llm($categories, $article_data) {
    $api_key = get_option('mosallas_api_key');
    $api_base = 'https://mosallas.ir/api/wordpress';
    
    if (empty($api_key)) {
        return null;
    }
    
    $request_data = array(
        'action' => 'select_category',
        'data' => array(
            'categories' => $categories,
            'articleTitle' => $article_data['title'],
            'articleContent' => $article_data['content'],
            'articleTopic' => $article_data['topic']
        )
    );
    
    $response = wp_remote_post($api_base . '/webhook', array(
        'headers' => array(
            'X-API-Key' => $api_key,
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode($request_data),
        'timeout' => 30
    ));
    
    if (is_wp_error($response)) {
        error_log('Mosallas LLM Category Selection API Error: ' . $response->get_error_message());
        return null;
    }
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    if (!$body || !isset($body['success']) || !$body['success']) {
        error_log('Mosallas LLM Category Selection API Error: ' . json_encode($body));
        return null;
    }
    
    return $body['category'] ?? null;
}
